/*
 * DisConstraintProblem.java
 *
 * 
 */
 package DisCSP.DistributedCSP;

import DisCSP.CSP.Constraint.*;
import DisCSP.CSP.*;
import DisCSP.DistributedCSP.ExternalConstraint.*;
import DisCSP.Exception.*;

/**
 * @ version 1.0
 *
 * @ author Nocerino Francesca
 *
 * @ since JDK 1.4
 *
 */


public class DisConstraintProblem
{
	private ConstraintProblem localCP;
	private ExtBinaryConstraintVector extConstr;
	

 	/** Costruttore di DisConstraintProblem
     * 
     * @param cp il CSP locale
     * 
     */	
	public DisConstraintProblem(ConstraintProblem cp)
	{
		localCP= cp;
		extConstr=new ExtBinaryConstraintVector();
	}
	
	/** Restituisce il numero di vincoli esterni del problema
	 *
     * @return il numero di vincoli esterni del DCP
     * 
     */								
	public int numberOfExternalConstraint()
	{
		return extConstr.size();
	}
	
	/** Restituisce l'i-esimo vincolo esterno del problema
	 *
	 * @param i l'indice del vincolo esterno
     * @return l'i-esimo vincolo esterno
     * 
     */			
	public ExtBinaryConstraint externalConstraintAt(int i)
	{
		return extConstr.externalConstraintAt(i);
	}

	/**  Permette di aggiungere un vincolo esterno al problema
	 *
	 * @param constr il vincolo esterno da aggiungere
     * 
     */					
	public void addExternalConstraint(ExtBinaryConstraint ext) throws UnknownVariableException
	{
		Variable app= ext.getLocalVariable();
		if(!localCP.isVarPresent(app))
			throw new UnknownVariableException(app.getName());
		extConstr.add(ext);
	}
		
	
	/** Restituisce il local CP associato al DCP
	 *
     * @return il local ConstraintProblem associato al DisConstraintProblem
     * 
     */		
		
	public ConstraintProblem localConstraintProblem()
	{
		return localCP;
	}

	
	/** Stampa il problema
	 *
     * @return una stringa che rappresenta il problema
     * 
     */			
	public String print()
	{
		String prob=localCP.print();
		
		prob+="\nExternal Constraints\n";
		
		for(int i=0;i<numberOfExternalConstraint();i++)
		{
			prob+=externalConstraintAt(i).print()+"\n";
		}

		return prob;
	}
	
	/** Permette di aggiungere una variabile al problema
	 *
	 * @param v la variabile
     * 
     */	
	public void addVariable(Variable v) throws ExistentVarException
	{
		localCP.addVariable(v);
	}
	
	/** Permette di rimuovere una variabile dal problema
	 *
	 * @param v la variabile
     * 
     */					
	public void removeVariable(Variable v)
	{
		localCP.removeVariable(v);
	}
	
	/**  Permette di aggiungere un vincolo interno al problema
	 *
	 * @param constr il vincolo esterno da aggiungere
	 * @param v la variabile esterna
     * 
     */	
	public void addConstraint(ExtBinaryConstraint constr, Variable v) throws UnknownVariableException, NotBinaryConstrException
	{
		if(constr.getClass().getName().equals("DisCSP.DistributedCSP.ExternalConstraint.ExtEqualConstraint") )
			localCP.addConstraint(new EqualConstraint(constr.getLocalVariable(),v));
			
		if(constr.getClass().getName().equals("DisCSP.DistributedCSP.ExternalConstraint.ExtNotEqualConstraint") )
			localCP.addConstraint(new NotEqualConstraint(constr.getLocalVariable(),v));
			
		if(constr.getClass().getName().equals("DisCSP.DistributedCSP.ExternalConstraint.ExtGreaterConstraint") )
			localCP.addConstraint(new GreaterConstraint(constr.getLocalVariable(),v));
			
		if(constr.getClass().getName().equals("DisCSP.DistributedCSP.ExternalConstraint.ExtGreaterEqualConstraint") )
			localCP.addConstraint(new GreaterEqualConstraint(constr.getLocalVariable(),v));
			
		if(constr.getClass().getName().equals("DisCSP.DistributedCSP.ExternalConstraint.ExtLessConstraint") )
			localCP.addConstraint(new GreaterConstraint(v,constr.getLocalVariable()));
			
		if(constr.getClass().getName().equals("DisCSP.DistributedCSP.ExternalConstraint.ExtLessEqualConstraint") )
			localCP.addConstraint(new GreaterEqualConstraint(v,constr.getLocalVariable()));			
	}


}